import BookPoster from "@/components/BookPoster";
import db from "./../../../../db";
import { Book, SimilarBook } from "./../../../../types";
import { notFound } from "next/navigation";

// refresh cache every 24 hours
export const revalidate = 60 * 60 * 24;

async function BookPage({
  params: { id },
}: {
  params: {
    id: string;
  };
}) {
  const books = db.collection("books");

  const search = await books.find(
    { $and: [{ _id: id }] },
    { projection: { $vector: 1 } }
  );

  if (!(await search.hasNext())) {
    return notFound();
  }

  const book = (await search.next()) as Book;

  const similarBooks = (await books
    .find(
      {
        Book_Title: { $ne: book?.Book_Title }, // Exclude the current book
      },
      {
        vector: book.$vector,
        limit: 10, // Fetch more results than needed to handle duplicates
        includeSimilarity: true,
      }
    )
    .toArray()) as SimilarBook[];

  // Deduplicate the results
  const seenIds = new Set();
  const deduplicatedBooks = similarBooks.filter((book) => {
    if (seenIds.has(book.$similarity)) {
      return false; // Skip duplicate
    }
    seenIds.add(book.$similarity);
    return true; // Keep unique book
  });

  // Limit to the desired number of results
  const finalBooks = deduplicatedBooks.slice(0, 5);

  return (
    <div>
      <div className="flex flex-col md:flex-row items-center gap-y-10 p-10 pb-0">
        {/* <Image
          src={book?.Image_URL_L}
          alt={book?.Book_Title}
          width={300}
          height={450}
          className="shrink-0 rounded-lg "
        /> */}
        <img
          src={book?.Image_URL_L}
          alt={book?.Book_Title}
          width={300}
          height={450}
        />
        <div className="px-2 md:px-10 flex flex-col gap-y-2">
          <h1 className="text-6xl font-bold">{book?.Book_Title}</h1>
          <p className="text-gray-600">{book?.Book_Author}</p>
          <p className="font-light">{book?.Publisher}</p>

          <div className="mt-auto grid grid-cols-2">
            <div className="font-semibold">
              <p>Authored by:</p>
              <p>Publisher:</p>
              <p>Year of Publication:</p>
            </div>
            <div>
              <p>{book?.Book_Author}</p>
              <p>{book?.Publisher}</p>
              <p>{book?.Year_Of_Publication}</p>
            </div>
          </div>
        </div>
      </div>

      <div className="">
        <h2 className="text-3xl pt-10 pl-10 font-bold ">
          Similar Books you may like
        </h2>
        <div className="flex justify-between items-center lg:flex-row gap-x-20 gap-y-10 pl-20 pr-10 py-10 overflow-x-scroll">
          {finalBooks.map((book, i) => (
            <BookPoster
              key={book._id}
              index={i + 1}
              //   similarityRating={Number(book.$similarity.toFixed(2)) * 100}
              book={book}
            />
          ))}
        </div>
      </div>
    </div>
  );
}

export default BookPage;
